#!/bin/bash

# ============================================================================
# Interactive Master Run Script
# Purpose: Provides an interactive menu to select and run any case folder
# ============================================================================

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Get the directory where this script is located (main project directory)
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
CASE_FOLDER_DIR="$SCRIPT_DIR/case_folder"

# Display header
clear
echo -e "${BLUE}╔════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║        OpenFOAM Hypersonic Flow Cases - Master Runner      ║${NC}"
echo -e "${BLUE}╚════════════════════════════════════════════════════════════╝${NC}"
echo ""

# Display menu
echo -e "${YELLOW}Available Cases:${NC}"
echo "─────────────────────────────────────"
echo -e "${GREEN}a${NC}) blockMesh     - Mesh generation using blockMesh utility"
echo -e "${GREEN}b${NC}) Mach-5        - Mach 5 flow analysis with ANSYS mesh"
echo -e "${GREEN}c${NC}) Mach-6        - Mach 6 flow analysis with ANSYS mesh"
echo -e "${GREEN}d${NC}) Mach-7        - Mach 7 flow analysis with ANSYS mesh"
echo -e "${GREEN}e${NC}) Mach-8        - Mach 8 flow analysis with ANSYS mesh"
echo "─────────────────────────────────────"
echo ""

# Get user input
read -p "Select a case (a-e): " user_input

# Validate and map input to case folder using case statement
case "$user_input" in
    a)
        CASE_NAME="blockMesh"
        ;;
    b)
        CASE_NAME="Mach-5"
        ;;
    c)
        CASE_NAME="Mach-6"
        ;;
    d)
        CASE_NAME="Mach-7"
        ;;
    e)
        CASE_NAME="Mach-8"
        ;;
    *)
        echo -e "${RED}✗ Invalid input. Please enter a letter from a to e.${NC}"
        exit 1
        ;;
esac

echo ""
echo -e "${BLUE}Selected case: ${GREEN}$CASE_NAME${NC}"
echo ""

# Navigate to the case folder
CASE_PATH="$CASE_FOLDER_DIR/$CASE_NAME"

if [ ! -d "$CASE_PATH" ]; then
    echo -e "${RED}✗ Error: Case folder '$CASE_PATH' not found!${NC}"
    exit 1
fi

# Change to case directory
cd "$CASE_PATH" || exit 1

echo -e "${YELLOW}Location: ${BLUE}$CASE_PATH${NC}"
echo ""

# Make all Allrun scripts executable
echo -e "${YELLOW}Setting executable permissions...${NC}"
chmod +x A* 2>/dev/null
echo -e "${GREEN}✓ Permissions set for A* files${NC}"
echo ""

# Display run information
echo -e "${BLUE}╔════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║                   STARTING SIMULATION                      ║${NC}"
echo -e "${BLUE}╚════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${YELLOW}Solver Configuration:${NC}"
echo "  • Turbulence Model: SST k-ω"
echo "  • Solver: rhoCentralFoam"
echo "  • Decomposition: Domain decomposition for parallel execution"
echo ""

# Run the Allrun script
if [ -f "$CASE_PATH/Allrun" ]; then
    echo -e "${GREEN}Executing Allrun script...${NC}"
    echo ""
    
    # Run with time tracking
    START_TIME=$(date +%s)
    ./Allrun
    EXIT_CODE=$?
    END_TIME=$(date +%s)
    ELAPSED=$((END_TIME - START_TIME))
    
    # Display completion message
    echo ""
    echo -e "${BLUE}╔════════════════════════════════════════════════════════════╗${NC}"
    if [ $EXIT_CODE -eq 0 ]; then
        echo -e "${BLUE}║${NC} ${GREEN}✓ Simulation completed successfully${NC}                    ${BLUE}║${NC}"
    else
        echo -e "${BLUE}║${NC} ${RED}✗ Simulation encountered errors (Exit Code: $EXIT_CODE)${NC} ${BLUE}║${NC}"
    fi
    echo -e "${BLUE}║${NC} Elapsed Time: ${YELLOW}$((ELAPSED / 60)) minutes $((ELAPSED % 60)) seconds${NC}              ${BLUE}║${NC}"
    echo -e "${BLUE}╚════════════════════════════════════════════════════════════╝${NC}"
    
    exit $EXIT_CODE
else
    echo -e "${RED}✗ Error: Allrun script not found in $CASE_PATH${NC}"
    exit 1
fi
